////////////////////////////////////////////////////////////////
// ReplaceOnSurveySavePlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.ReplaceOnSurveySavePlugin;


import com.objectplanet.survey.event.Event;
import com.objectplanet.survey.event.EventBusManager;
import com.objectplanet.survey.event.IEventListener;
import com.objectplanet.survey.event.events.SurveySaveEvent;
import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.PluginConstants;
import com.objectplanet.survey.plugin.ProcessResult;
import com.objectplanet.survey.plugin.api.PluginUtil;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Properties;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;


public class ReplaceOnSurveySavePlugin extends Plugin implements IEventListener {

	private static final String	PARAM_REPLACE_WHAT			= "plugin_replaceWhat";
	private static final String	PARAM_REPLACE_WITH			= "plugin_replaceWith";
	private static final String	PROPERTY_EMAIL_REPLACE_WITH	= "email.replaceWith";
	private static final String	PROPERTY_EMAIL_REPLACE_WHAT	= "email.replaceWhat";

	// attribute for setup screen validation errors.
	private static final String	VALIDATION_FAILED			= "validationFailed";

	Properties					props;
	File						propsFile;

	String						replaceWhat;
	String						replaceWith;


	public ReplaceOnSurveySavePlugin() {
		super();
		props = new Properties();
		propsFile = new File(PluginUtil.getSystemHome() + "/WEB-INF/plugins/replaceonsurveysave.properties");
	}


	/**
	 * Returns html for the setup page
	 *
	 * @see Plugin#getSetupHTML(HashMap)
	 */
	public String getSetupHTML(HashMap resources) {
		HttpServletRequest request = (HttpServletRequest) resources.get(PluginConstants.RESOURCE_HTTP_REQUEST);

		// find out if we returned to the setup page after validation failure.
		Boolean validationFailedAttribute = (Boolean) request.getAttribute(VALIDATION_FAILED);
		boolean validationFailed = validationFailedAttribute != null && validationFailedAttribute.booleanValue();

		// get values to put in the form fields
		String formReplaceWhat = validationFailed ? request.getParameter(PARAM_REPLACE_WHAT) : replaceWhat;
		String formReplaceWith = validationFailed ? request.getParameter(PARAM_REPLACE_WITH) : replaceWith;

		StringBuffer html = new StringBuffer();
		html.append("<table width=\"100%\" class=sectionTable>");

		// Plugin description
		html.append("<tr><td class=form colspan=2>");
		html.append("This plugin replaces some text in the email sent to the respondent on survey save action. <br><br>");
		html.append("Applies to all surveys with \"allow save and return\" feature turned on. ");
		html.append("</td></tr><tr><td class=listitem colspan=2 height=10></td></tr>");

		// Text to replace
		html.append("<tr><td class='form label' width=\"30%\">Text to replace:&nbsp;</td>");
		html.append("	 <td class='form value' width=\"70%\">");
		html.append("		<input name=\"" + PARAM_REPLACE_WHAT).append("\" class=width200 value=\"").append(getStringValue(formReplaceWhat)).append("\"> (regular expression)");
		html.append("	 </td></tr>");

		// New text value
		html.append("<tr><td class='form label'>Replace with:&nbsp;</td>");
		html.append("	 <td class='form value'>");
		html.append("		<input name=\"").append(PARAM_REPLACE_WITH).append("\" class=width200 value=\"").append(getStringValue(formReplaceWith)).append("\">");
		html.append("</td></tr>");

		// save button
		html.append("<tr><td></td><td class=buttons>");
		html.append("	<INPUT TYPE=submit class=button name=submitButton value=\"Save\">");
		html.append("		</td>");
		html.append("	</tr>");
		html.append("</table>");
		return html.toString();
	}


	/**
	 * Method for processing the setup screen. The values from the setup screen is collected by
	 * Opinio and passed to this method in a Map. It is up to the implementor (developer) of this
	 * method to process the values appropriately, and to persist the values if necessary.
	 *
	 * @param values The values entered by the user in the setup screen.
	 * @return The response message, with info about the result of setup values processing.
	 */
	public ProcessResult processSetupHTML(HashMap resources) {
		HttpServletRequest request = (HttpServletRequest) resources.get(PluginConstants.RESOURCE_HTTP_REQUEST);

		// get values to put in the form fields
		String formReplaceWhat = request.getParameter(PARAM_REPLACE_WHAT);
		String formReplaceWith = request.getParameter(PARAM_REPLACE_WITH);

		// validate input
		if (formReplaceWhat == null || formReplaceWhat.trim().equals("")) {
			request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
			return new ProcessResult("Please enter \"Text to replace\"", true);
		} else {
			// validate the reg exp
			try {
				Pattern.compile(formReplaceWhat);
			} catch (Exception e) {
				request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
				return new ProcessResult("Invalid pattern. " + e.getMessage(), true);
			}
		}

		// set the plugin properties
		props.setProperty(PROPERTY_EMAIL_REPLACE_WHAT, formReplaceWhat);
		props.setProperty(PROPERTY_EMAIL_REPLACE_WITH, formReplaceWith);

		// save the properties (make them persistent)
		try {
			PluginUtil.storeProps(props, propsFile);
		} catch (Exception e) {
			request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
			return new ProcessResult("Unable to store the properties:<br><br>" + e.getMessage(), true);
		}

		// set the plugin attributes as class fields for faster access
		setAttributes();

		return new ProcessResult("Plugin settings have been successfully stored.", true);
	}


	/**
	 * Starts the plugin. The survey event bus registrations are made here. Any defaults, plus
	 * whatever event settings are made through the setup screen are read here.
	 */
	public void start() {
		// get the properties file and store the values in a local properties
		// variable.
		try {
			FileInputStream in = new FileInputStream(propsFile);
			BufferedInputStream buf = new BufferedInputStream((InputStream) in);
			props.load(buf);
			in.close();
		} catch (FileNotFoundException fe) {
			PluginUtil.warn(propsFile.getAbsolutePath() + " file not found: " + fe.toString());
			return;
		} catch (IOException ie) {
			PluginUtil.error(propsFile.getAbsolutePath() + "/ReplaceOnSurveySave.properties file could not be loaded: " + ie.getMessage());
			return;
		}

		setAttributes();

		EventBusManager eventMgr = EventBusManager.instance();
		eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_SURVEY_SAVE);

		PluginUtil.info("ReplaceOnSurveySavePlugin loaded");
	}


	/**
	 * @see IEventListener#handleEvent(Event)
	 */
	public void handleEvent(Event event) {
		if (event instanceof SurveySaveEvent) {
			SurveySaveEvent ssEvent = (SurveySaveEvent) event;
			if (replaceWhat != null && !replaceWhat.trim().equals("")) {
				try {
					String originalMessage = ssEvent.getSurveyReturnMessage();
					String modifiedMessage = originalMessage.replaceAll(replaceWhat, replaceWith);
					ssEvent.setSurveyReturnMessage(modifiedMessage);
				} catch (Exception e) {
					PluginUtil.error("ReplaceOnSurveySavePlugin: unable to replace text " + replaceWhat + " with " + replaceWith + ": " + e.getMessage());
				}
			}
		}
	}


	/**
	 * Sets the attributes attribute of the ResponseNotifierPlugin object
	 */
	private void setAttributes() {
		replaceWhat = props.getProperty(PROPERTY_EMAIL_REPLACE_WHAT);
		replaceWith = props.getProperty(PROPERTY_EMAIL_REPLACE_WITH);
	}


	private String getStringValue(String value) {
		if (value == null) {
			return "";
		}

		// replace <, >, " with HTML code
		value = value.replaceAll("<", "&lt;");
		value = value.replaceAll(">", "&gt;");
		value = value.replaceAll("\"", "&quot;");
		return value;
	}
}
