////////////////////////////////////////////////////////////////
// OtherFieldRequiredPlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.OtherFieldRequiredPlugin;


import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.api.HtmlUtils;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.QuestionMultiple;
import com.objectplanet.survey.plugin.api.Respondent;
import com.objectplanet.survey.plugin.api.Response;
import com.objectplanet.survey.plugin.api.ResponseMultiple;
import com.objectplanet.survey.plugin.api.Survey;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.api.SurveySecurityException;
import com.objectplanet.survey.plugin.api.SurveySystemException;
import com.objectplanet.survey.plugin.interfaces.IPluginValidator;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.HashMap;


/**
 * This plugin validates the other field in a multiple choice if there is one. The other field is
 * made required and empty fields is not allowed. There is no advanced setup screen and it is
 * possible to add a customized error message (shown on the simple setup next to the plugin link).
 *
 * @author Ina Skåre
 *
 */
public class OtherFieldRequiredPlugin extends Plugin implements IPluginValidator {

	private final static String	ERROR_MSG_INPUT_NAME	= "OtherFieldRequiredPlugin_errorMessage";

	// static default value for error message
	private String				ERROR_MSG_DEFAULT		= "Please specify \"Other\" value.";


	/**
	 * Returns whether the plugin can validate the type of validation, not
	 *
	 * @param customValidatorType The type of custom validation
	 * @return Whether the plugin can validate or not
	 */
	public boolean canValidate(int customValidatorType) {
		return (customValidatorType == CUSTOM_VALIDATOR_QUESTION);
	}


	/**
	 * Validate the other field
	 *
	 * @param sId Survey id
	 * @param qNo Question number
	 * @param respId Respondent id
	 * @param settings Attributes and values settings
	 * @return Returns if valid or not
	 */
	public boolean validate(long sId, int qNo, long respId, HashMap settings) {
		try {

			// get respondent object
			Respondent respondent = SurveyManager.getRespondentForSurvey(respId, sId);

			// get response object for the question
			Response responseToValidate = respondent.getResponse(qNo);

			// if multiple choice question
			if (responseToValidate instanceof ResponseMultiple) {

				// get question
				Survey survey = SurveyManager.getSurvey(PluginUtil.getSystemUser(), sId);
				QuestionMultiple question = (QuestionMultiple) survey.getQuestion(qNo);

				boolean hasOtherField = question.getOtherField();
				if (!hasOtherField) {
					// If no other field, there is nothing to validate, return true.
					return true;
				}

				// check if last option is selected
				int optionCount = question.getOptionCount();
				ResponseMultiple multipleResponse = (ResponseMultiple) responseToValidate;
				boolean lastOptionSelected = multipleResponse.getOptionValue(optionCount - 1);
				if (!lastOptionSelected) {
					// If last option is not selected, do not validate
					return true;
				}
				// validate other field
				String otherValue = multipleResponse.getOtherValue();
				if ((otherValue == null) || (otherValue.trim().equals(""))) {
					return false;
				}
			}

		} catch (SurveySystemException ssex) {
			PluginUtil.error("FreeTextRequiredPlugin.validate(): Unable to validate response (sId " + sId + ", qNo " + qNo + ", respId " + respId + "): " + getStackTraceAsString(ssex));
			// can't get the respondent, we can't validate and return true
		} catch (SurveySecurityException se) {
			PluginUtil.error("FreeTextRequiredPlugin.validate(): Unable to validate response (sId " + sId + ", qNo " + qNo + ", respId " + respId + "): " + getStackTraceAsString(se));
			// can't get the respondent, we can't validate and return true
		}

		return true;
	}


	/**
	 * Returns the simple HTML setup to be placed in the question edit screen. This code will be
	 * placed within a html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getSimpleHTML(HashMap settings) {
		String errorMsgInputValue = getSetting(settings, ERROR_MSG_INPUT_NAME);
		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr><td width='50%'></td>");
		sb.append("<td class='form label' align=right>");
		sb.append("Custom error message:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(ERROR_MSG_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(errorMsgInputValue)).append("\" class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Gets the errorMessage to show from the settings hashmap
	 *
	 * @param settings Attributes and values settings
	 * @return The errorMessage value
	 */
	public String getErrorMessage(HashMap settings) {
		String errorMessageToReturn = getSetting(settings, ERROR_MSG_INPUT_NAME);
		if (errorMessageToReturn.equals("")) {
			errorMessageToReturn = ERROR_MSG_DEFAULT;
		}
		return errorMessageToReturn;
	}


	/**
	 * Gets the javaScriptFunctionCall name for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunctionCall name
	 */
	public String getJavaScriptFunctionCall(int questionNo, HashMap settings) {
		StringBuffer sb = new StringBuffer();
		sb.append("checkOtherFieldRequired(form.");
		sb.append(HtmlUtils.getMultipleOtherInputName(questionNo));
		sb.append(")");
		return sb.toString();
	}


	/**
	 * Gets the javaScriptFunction code for client side validation. TODO: implement
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunction code
	 */
	public String getJavaScriptFunction(int questionNo, HashMap settings) {
		StringBuffer sb = new StringBuffer();
		sb.append("function checkOtherFieldRequired(field) {\n");
		sb.append("	return true;\n");
		sb.append("}\n");
		return sb.toString();
	}


	/**
	 * Gets a specific setting value
	 *
	 * @param settings Attributes and values settings
	 * @param parameter Parameter name
	 *
	 * @return The javaScriptFunction code
	 */
	private String getSetting(HashMap settings, String parameter) {
		String setting = "";
		if (settings != null) {
			setting = (String) settings.get(parameter);
			if (setting == null) {
				setting = "";
			}
		}
		return setting;
	}


	/**
	 * Gets the stackTraceAsString attribute
	 *
	 * @param throwable Throwable, containing the stack trace.
	 * @return The stackTraceAsString value
	 */
	private static String getStackTraceAsString(Throwable throwable) {
		ByteArrayOutputStream ostr = new ByteArrayOutputStream();
		throwable.printStackTrace(new PrintStream(ostr));
		return ostr.toString();
	}


	/*
	 * (non-Javadoc)
	 *
	 * @see com.objectplanet.survey.plugin.interfaces.IPluginValidator#getAdvancedHTML(java.util.HashMap)
	 */
	public String getAdvancedHTML(HashMap settings) {
		// TODO Auto-generated method stub
		return null;
	}

}
