////////////////////////////////////////////////////////////////
// MaxPercentagePlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.MaxPercentagePlugin;


import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.api.HtmlUtils;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.Question;
import com.objectplanet.survey.plugin.api.QuestionMatrix;
import com.objectplanet.survey.plugin.api.Respondent;
import com.objectplanet.survey.plugin.api.ResponseMatrix;
import com.objectplanet.survey.plugin.api.Survey;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.interfaces.IPluginValidator;
import java.util.HashMap;


/**
 * This plugin counts responses in all matrix cells and fail the validation if sum of all cells are
 * bigger than the defined total. Only cells of type Numeric integer are summarized.
 *
 * @author Rikard Halvorsen
 * @created April 04, 2005
 */
public class MaxPercentagePlugin extends Plugin implements IPluginValidator {

	// static values for input fields
	private final static String	MAX_PERCENTAGE			= "MaxPercentagePlugin_maxPercentage";
	private final static String	ERROR_MSG_INPUT_NAME	= "MaxPercentagePlugin_errorMessage";

	// static default value for error message
	private String				ERROR_MSG_DEFAULT		= "The max percentage limit has been exceeded";


	public String getHelpHTML() {
		return "This plugin counts matrix cells responses and validated the sum against the defined total. Only cells of type Numeric integer are summarized.";
	}


	/**
	 * Returns whether the plugin can validate the type of validation, not
	 *
	 * @param customValidatorType The type of custom validation
	 * @return Whether the plugin can validate or not
	 */
	public boolean canValidate(int customValidatorType) {
		return (customValidatorType == CUSTOM_VALIDATOR_QUESTION);
	}


	/**
	 * Validates the question based on the attributes and values set in the settings hashmap
	 *
	 * @param sId Survey id
	 * @param qNo Question number
	 * @param respId Respondent id
	 * @param settings Attributes and values settings
	 * @return Returns if valid or not
	 */
	public boolean validate(long sId, int qNo, long respId, HashMap settings) {
		// ignore the validation if the settings are not set
		if (settings == null) {
			return true;
		}

		try {
			// get the survey object
			Survey survey = SurveyManager.getSurvey(PluginUtil.getSystemUser(), sId);

			// this plugin can only validate matrix question types.
			Question question = survey.getQuestion(qNo);
			if (question.getQuestionType() != Question.QUESTION_MATRIX) {
				return true;
			}

			// get the question object
			QuestionMatrix matrixQuestion = (QuestionMatrix) question;

			// get respondent object
			Respondent respondent = SurveyManager.getRespondentForSurvey(respId, sId);

			// get response object for the question
			ResponseMatrix responseMatrixToValidate = (ResponseMatrix) respondent.getResponse(qNo);

			// ignore all validation if responseToValidate is null
			if (responseMatrixToValidate == null) {
				return true;
			}

			// get the value for the max percentage
			int maxPercentage = -1;
			try {
				String maxPercentageSetting = getSetting(settings, MAX_PERCENTAGE);
				PluginUtil.debug("MaxPercentagePlugin.validate(): maxPercentageSetting=" + maxPercentageSetting);
				maxPercentage = Integer.parseInt(maxPercentageSetting);
			} catch (Exception nfex) {
				PluginUtil.error("MaxPercentagePlugin.validate(): Unable to parse maxPercentageSetting: " + nfex.toString());
				return true;
			}

			// ignore max percentage validation if the maxPercentage isn't set properly
			if (maxPercentage < 0) {
				return true;
			}
			// validate the max percentage, if maxPercentage exceeded set isValid = false
			int totalPercentage = 0;
			int rowCount = matrixQuestion.getRowCount();
			int columnCount = matrixQuestion.getColumnCount();

			// count the sum of all cells
			for (int row = 0; row < rowCount; row++) {
				for (int col = 0; col < columnCount; col++) {
					int cellType = matrixQuestion.getCellType(col, row);
					if (cellType == QuestionMatrix.CELL_NUMERIC_INT) {
						String numericValue = responseMatrixToValidate.getCellValue(col, row, cellType);

						if (numericValue != null && !numericValue.trim().equals("")) {
							totalPercentage += Integer.parseInt(numericValue);
						}
					}
				}
			}

			if (totalPercentage > maxPercentage) {
				PluginUtil.debug("MaxPercentagePlugin.validate(): totalPercentage (" + totalPercentage + ") is bigger than maxPercentage (" + maxPercentage + ")");
				return false;
			} else {
				PluginUtil.debug("MaxPercentagePlugin.validate(): totalPercentage (" + totalPercentage + ") is NOT bigger than maxPercentage (" + maxPercentage + ")");
			}
		} catch (Exception ex) {
			PluginUtil.error("MaxPercentagePlugin.validate(): Unable to validate response: " + ex.toString());
			// we can't validate and return true
			return true;
		}
		return true;
	}


	/**
	 * Returns the simple HTML setup to be placed in the question edit screen. This code will be
	 * placed within a html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getSimpleHTML(HashMap settings) {
		String maxPercentageInputValue = getSetting(settings, MAX_PERCENTAGE);
		String errorMsgInputValue = getSetting(settings, ERROR_MSG_INPUT_NAME);

		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr>");
		sb.append("<td width=\"50%\" class='form label'>");
		sb.append("Max percentage:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(MAX_PERCENTAGE);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(maxPercentageInputValue)).append("\" class=width50></td>");
		sb.append("<td width=\"50%\" class='form label' align=right>");
		sb.append("Custom error message:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(ERROR_MSG_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(errorMsgInputValue)).append("\" class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Returns the advanced HTML setup to be placed in a popup window when user click on link in the
	 * question edit screen. This code will be placed within an html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getAdvancedHTML(HashMap settings) {
		return null;
	}


	/**
	 * Gets the errorMessage to show from the settings hashmap
	 *
	 * @param settings Attributes and values settings
	 * @return The errorMessage value
	 */
	public String getErrorMessage(HashMap settings) {
		String errorMessageToReturn = getSetting(settings, ERROR_MSG_INPUT_NAME);
		if (errorMessageToReturn.equals("")) {
			errorMessageToReturn = ERROR_MSG_DEFAULT;
		}
		return errorMessageToReturn;
	}


	/**
	 * Gets the javaScriptFunctionCall name for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunctionCall name
	 */
	public String getJavaScriptFunctionCall(int questionNo, HashMap settings) {
		// not supported in this version of the MaxPercentagePlugin
		return null;
	}


	/**
	 * Gets the javaScriptFunction code for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunction code
	 */
	public String getJavaScriptFunction(int questionNo, HashMap settings) {
		// not supported in this version of the MaxPercentagePlugin
		return null;
	}


	/**
	 * Gets a specific setting value
	 *
	 * @param settings Attributes and values settings
	 * @param parameter Parameter name
	 *
	 * @return The javaScriptFunction code
	 */
	private String getSetting(HashMap settings, String parameter) {
		String setting = "";
		if (settings != null) {
			setting = (String) settings.get(parameter);
			if (setting == null) {
				setting = "";
			}
		}
		return setting;
	}
}
