////////////////////////////////////////////////////////////////
// MatrixSumValidatorPlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.MatrixSumValidatorPlugin;


import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.api.HtmlUtils;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.Question;
import com.objectplanet.survey.plugin.api.QuestionMatrix;
import com.objectplanet.survey.plugin.api.Respondent;
import com.objectplanet.survey.plugin.api.ResponseMatrix;
import com.objectplanet.survey.plugin.api.Survey;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.interfaces.IPluginValidator;

import java.util.HashMap;


/**
 * This plugin summarize responses in all matrix cells and perform validation of the sum againts th
 * plugin settings. Only cells of type Numeric integer and decimal are summarized.
 *
 * @author Rikard Halvorsen
 * @created September 05, 2005
 */
public class MatrixSumValidatorPlugin extends Plugin implements IPluginValidator {

	// static values for input fields and type of validation
	private final static String	SUM						= "MatrixSumValidatorPlugin_sum";
	private final static String	TYPE					= "MatrixSumValidatorPlugin_type";
	private final static int	MIN						= 1;
	private final static int	CONSTANT				= 2;
	private final static int	MAX						= 3;
	private final static String	ERROR_MSG_INPUT_NAME	= "MatrixSumValidatorPlugin_errorMessage";

	// static default value for error message
	private String				ERROR_MSG_DEFAULT		= "Sum is not correct";


	/**
	 * Returns whether the plugin can validate the type of validation, not
	 *
	 * @param customValidatorType The type of custom validation
	 * @return Whether the plugin can validate or not
	 */
	public boolean canValidate(int customValidatorType) {
		return (customValidatorType == CUSTOM_VALIDATOR_QUESTION);
	}


	/**
	 * Validates the question based on the attributes and values set in the settings hashmap
	 *
	 * @param sId Survey id
	 * @param qNo Question number
	 * @param respId Respondent id
	 * @param settings Attributes and values settings
	 * @return Returns if valid or not
	 */
	public boolean validate(long sId, int qNo, long respId, HashMap settings) {
		// ignore the validation if the settings is null
		if (settings == null) {
			return true;
		}

		try {
			// get the survey object
			Survey survey = SurveyManager.getSurvey(PluginUtil.getSystemUser(), sId);

			// this plugin can only validate matrix question types.
			Question question = survey.getQuestion(qNo);
			if (question.getQuestionType() != Question.QUESTION_MATRIX) {
				return true;
			}

			// get the question object
			QuestionMatrix matrixQuestion = (QuestionMatrix) question;

			// get respondent object
			Respondent respondent = SurveyManager.getRespondentForSurvey(respId, sId);

			// get response object for the question
			ResponseMatrix responseMatrixToValidate = (ResponseMatrix) respondent.getResponse(qNo);

			// ignore all validation if responseToValidate is null
			if (responseMatrixToValidate == null) {
				return true;
			}

			// get the value for the sum and type
			double sum = Double.MIN_VALUE;
			int type = Integer.MIN_VALUE;

			try {
				String sumSetting = getSetting(settings, SUM);
				PluginUtil.debug("MatrixSumValidatorPlugin.validate(): sumSetting=" + sumSetting);
				sumSetting = sumSetting.replaceAll(",", ".");
				sum = new Double(sumSetting).doubleValue();
			} catch (Exception nfex) {
				PluginUtil.error("MatrixSumValidatorPlugin.validate(): Unable to parse sumSetting: " + nfex.toString());
			}

			try {
				String typeSetting = getSetting(settings, TYPE);
				PluginUtil.debug("MatrixSumValidatorPlugin.validate(): typeSetting=" + typeSetting);
				type = new Integer(typeSetting).intValue();
			} catch (Exception nex) {
				PluginUtil.error("MatrixSumValidatorPlugin.validate(): Unable to parse typeSetting: " + nex.toString());
			}

			// ignore sum validation if the sum or type isn't set properly
			if (sum != Double.MIN_VALUE && type != Integer.MIN_VALUE) {

				double accumulatedSum = 0;

				int rowCount = matrixQuestion.getRowCount();
				int columnCount = matrixQuestion.getColumnCount();

				for (int row = 0; row < rowCount; row++) {
					for (int col = 0; col < columnCount; col++) {
						int cellType = matrixQuestion.getCellType(col, row);
						if (cellType == QuestionMatrix.CELL_NUMERIC_INT || cellType == QuestionMatrix.CELL_NUMERIC_DEC) {
							String numericValue = responseMatrixToValidate.getCellValue(col, row, cellType);
							if (numericValue != null && !numericValue.trim().equals("")) {
								accumulatedSum += new Double(numericValue).doubleValue();
							}
						}
					}
				}

				switch (type) {
					case (MIN):
						// if accumulatedSum is lower than sum set isValid = false
						if (accumulatedSum < sum) {
							PluginUtil.debug("MatrixSumValidatorPlugin.validate(): accumulated sum (" + accumulatedSum + ") is lower than sum (" + sum + ")");
							return false;
						} else {
							PluginUtil.debug("MatrixSumValidatorPlugin.validate(): accumulated sum (" + accumulatedSum + ") is lower than sum (" + sum + ")");
							return true;
						}
					case (CONSTANT):
						// if accumulatedSum is not the same as sum set isValid = false
						if (accumulatedSum != sum) {
							PluginUtil.debug("MatrixSumValidatorPlugin.validate(): accumulated sum (" + accumulatedSum + ") is not equal to sum (" + sum + ")");
							return false;
						} else {
							PluginUtil.debug("MatrixSumValidatorPlugin.validate(): accumulated sum (" + accumulatedSum + ") is equal to sum (" + sum + ")");
							return true;
						}
					case (MAX):
						// if accumulatedSum is bigger than sum set isValid = false
						if (accumulatedSum > sum) {
							PluginUtil.debug("MatrixSumValidatorPlugin.validate(): accumulated sum (" + accumulatedSum + ") is bigger than sum (" + sum + ")");
							return false;
						} else {
							PluginUtil.debug("MatrixSumValidatorPlugin.validate(): accumulated sum (" + accumulatedSum + ") is bigger than sum (" + sum + ")");
							return true;
						}
				}
			}
		} catch (Exception ex) {
			PluginUtil.error("MatrixSumValidatorPlugin.validate(): Unable to validate response: " + ex.toString());
			// we can't validate and return true
			return true;
		}

		return true;
	}


	/**
	 * Returns the simple HTML setup to be placed in the question edit screen. This code will be
	 * placed within a html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getSimpleHTML(HashMap settings) {
		String sumInputValue = getSetting(settings, SUM);
		String typeInputValue = getSetting(settings, TYPE);
		String errorMsgInputValue = getSetting(settings, ERROR_MSG_INPUT_NAME);

		int type = MIN;

		try {
			type = new Integer(typeInputValue).intValue();
		} catch (NumberFormatException nfex) {
			/* do nothing, default type will be selected */
		} catch (NullPointerException nex) {
			/* do nothing, default type will be selected */
		}

		String minSelected = "";
		String constantSelected = "";
		String maxSelected = "";
		switch (type) {
			case (MIN):
				minSelected = " selected";
				break;
			case (CONSTANT):
				constantSelected = " selected";
				break;
			case (MAX):
				maxSelected = " selected";
				break;
		}

		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr>");
		sb.append("<td width=\"25%\" class='form label'>");
		sb.append("Sum:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(SUM);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(sumInputValue)).append("\" class=width50>");
		sb.append("</td><td width=\"25%\"  class='form label'>");
		sb.append("Type:&nbsp;");
		sb.append("<select class=select name=\"").append(TYPE).append("\">");
		sb.append("	<option value=\"").append(MIN).append("\"").append(minSelected).append(">Min</option>");
		sb.append("	<option value=\"").append(MAX).append("\"").append(maxSelected).append(">Max</option>");
		sb.append("	<option value=\"").append(CONSTANT).append("\"").append(constantSelected).append(">Constant</option>");
		sb.append("</select>");
		sb.append("</td>");
		sb.append("<td width=\"50%\"  class='form label' align=right>");
		sb.append("Custom error message:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(ERROR_MSG_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(errorMsgInputValue)).append("\" class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Returns the advanced HTML setup to be placed in a popup window when user click on link in the
	 * question edit screen. This code will be placed within an html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getAdvancedHTML(HashMap settings) {
		return null;
	}


	/**
	 * Gets the errorMessage to show from the settings hashmap
	 *
	 * @param settings Attributes and values settings
	 * @return The errorMessage value
	 */
	public String getErrorMessage(HashMap settings) {
		String errorMessageToReturn = getSetting(settings, ERROR_MSG_INPUT_NAME);
		if (errorMessageToReturn.equals("")) {
			errorMessageToReturn = ERROR_MSG_DEFAULT;
		}
		return errorMessageToReturn;
	}


	/**
	 * Gets the javaScriptFunctionCall name for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunctionCall name
	 */
	public String getJavaScriptFunctionCall(int questionNo, HashMap settings) {
		// not supported in this version of the MaxPercentagePlugin
		return null;
	}


	/**
	 * Gets the javaScriptFunction code for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunction code
	 */
	public String getJavaScriptFunction(int questionNo, HashMap settings) {
		// not supported in this version of the MaxPercentagePlugin
		return null;
	}


	/**
	 * Gets a specific setting value
	 *
	 * @param settings Attributes and values settings
	 * @param parameter Parameter name
	 *
	 * @return The javaScriptFunction code
	 */
	private String getSetting(HashMap settings, String parameter) {
		String setting = "";
		if (settings != null) {
			setting = (String) settings.get(parameter);
			if (setting == null) {
				setting = "";
			}
		}
		return setting;
	}
}
