////////////////////////////////////////////////////////////////
// FreeTextValidatorPlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.FreeTextValidatorPlugin;


import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.HtmlUtils;
import com.objectplanet.survey.plugin.interfaces.IPluginValidator;
import com.objectplanet.survey.plugin.api.Respondent;
import com.objectplanet.survey.plugin.api.Response;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.api.SurveySystemException;

import java.util.HashMap;


/**
 * Implements FreeTextValidatorPlugin functionality.
 *
 * @author Rikard Halvorsen
 * @created September 27, 2004
 */
public class FreeTextValidatorPlugin extends Plugin implements IPluginValidator {

	// static values for input fields
	private final static String	MAX_LENGTH_INPUT_NAME		= "FreeTextValidatorPlugin_freeTextMaxLength";
	private final static String	WORD_REQUIRED_INPUT_NAME	= "FreeTextValidatorPlugin_wordRequired";
	private final static String	ERROR_MSG_INPUT_NAME		= "FreeTextValidatorPlugin_errorMessage";

	// static default value for error message
	private String				ERROR_MSG_DEFAULT			= "The free text max length has been exceeded and/or the specified word is missing";


	/**
	 * Returns whether the plugin can validate the type of validation, not
	 *
	 * @param customValidatorType The type of custom validation
	 * @return Whether the plugin can validate or not
	 */
	public boolean canValidate(int customValidatorType) {
		return (customValidatorType == CUSTOM_VALIDATOR_QUESTION);
	}


	/**
	 * Validates the question based on the attributes and values set in the settings hashmap
	 *
	 * @param sId Survey id
	 * @param qNo Question number
	 * @param respId Respondent id
	 * @param settings Attributes and values settings
	 * @return Returns if valid or not
	 */
	public boolean validate(long sId, int qNo, long respId, HashMap settings) {
		// ignore the validation if the settings is null
		if (settings == null) {
			return true;
		}

		boolean isValid = true;

		try {
			// get respondent object
			Respondent respondent = SurveyManager.getRespondentForSurvey(respId, sId);

			// get response object for the question
			Response responseToValidate = respondent.getResponse(qNo);

			// get the free text response value
			String freeTextValue = responseToValidate.getFreeText();

			// ignore all validation if free text value is null
			if (freeTextValue != null && !freeTextValue.equals("")) {
				// get the value for the free text max length
				String maxLengthInputValue = getSetting(settings, MAX_LENGTH_INPUT_NAME);
				int maxLength = -1;

				try {
					maxLength = Integer.parseInt(maxLengthInputValue);
				} catch (NumberFormatException nfex) {
					PluginUtil.debug("FreeTextValidatorPlugin.validate(): Unable to parse maxLengthInputValue: " + nfex.toString());
				} catch (NullPointerException nex) {
					PluginUtil.debug("FreeTextValidatorPlugin.validate(): Unable to parse maxLengthInputValue: " + nex.toString());
				}

				// ignore max length validation if the maxLength isn't set properly
				if (maxLength > -1) {
					// validate the free text length, if max length exceeded set isValid = false
					if (freeTextValue.length() > maxLength) {
						isValid = false;
					}
				}

				// validate word required
				String wordRequiredInputValue = getSetting(settings, WORD_REQUIRED_INPUT_NAME).toLowerCase();

				// ignore word required validation if value is empty
				if (!wordRequiredInputValue.equals("")) {
					// validate the free text word required, if word not included set isValid =
					// false
					if (freeTextValue.toLowerCase().indexOf(wordRequiredInputValue) == -1) {
						isValid = false;
					}
				}
			}
		} catch (SurveySystemException ssex) {
			PluginUtil.error("FreeTextValidatorPlugin.validate(): Unable to validate response: " + ssex.toString());
			// can't get the respondent, we can't validate and return true
		}
		return isValid;
	}


	/**
	 * Returns the simple HTML setup to be placed in the question edit screen. This code will be
	 * placed within a html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getSimpleHTML(HashMap settings) {
		String maxLengthInputValue = getSetting(settings, MAX_LENGTH_INPUT_NAME);
		String errorMsgInputValue = getSetting(settings, ERROR_MSG_INPUT_NAME);

		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr>");
		sb.append("<td width=\"50%\" class=\"form label\">");
		sb.append("Free text max length:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(MAX_LENGTH_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(maxLengthInputValue)).append("\" class=width50>&nbsp;");
		sb.append("</td>");
		sb.append("<td width=\"50%\"  class=\"form label\" align=right>");
		sb.append("Custom error message:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(ERROR_MSG_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(errorMsgInputValue)).append("\" class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Returns the advanced HTML setup to be placed in a popup window when user click on link in the
	 * question edit screen. This code will be placed within an html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getAdvancedHTML(HashMap settings) {
		String maxLengthInputValue = getSetting(settings, MAX_LENGTH_INPUT_NAME);
		String wordRequiredInputValue = getSetting(settings, WORD_REQUIRED_INPUT_NAME);
		String errorMsgInputValue = getSetting(settings, ERROR_MSG_INPUT_NAME);

		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr>");
		sb.append("<td width=\"25%\"  class=\"form label\">");
		sb.append("Free text max length: ");
		sb.append("</td>");
		sb.append("<td width=\"75%\"  class=\"form value\">");
		sb.append("<INPUT TYPE=text name=\"").append(MAX_LENGTH_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(maxLengthInputValue)).append("\" size=5 class=width30>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("<tr>");
		sb.append("<td class=\"form label\">");
		sb.append("Word required: </td><td class=\"form value\">");
		sb.append("<INPUT TYPE=text name=\"").append(WORD_REQUIRED_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(wordRequiredInputValue)).append("\" size=5 class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("<tr>");
		sb.append("<td class=\"form label\">");
		sb.append("Custom error message:&nbsp;</td><td class=\"form value\">");
		sb.append("<INPUT TYPE=text name=\"").append(ERROR_MSG_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(errorMsgInputValue)).append("\" size=30 class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Gets the errorMessage to show from the settings hashmap
	 *
	 * @param settings Attributes and values settings
	 * @return The errorMessage value
	 */
	public String getErrorMessage(HashMap settings) {
		String errorMessageToReturn = getSetting(settings, ERROR_MSG_INPUT_NAME);
		if (errorMessageToReturn.equals("")) {
			errorMessageToReturn = ERROR_MSG_DEFAULT;
		}
		return errorMessageToReturn;
	}


	/**
	 * Gets the javaScriptFunctionCall name for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunctionCall name
	 */
	public String getJavaScriptFunctionCall(int questionNo, HashMap settings) {
		String maxLengthInputValue = getSetting(settings, MAX_LENGTH_INPUT_NAME);
		String wordRequiredInputValue = getSetting(settings, WORD_REQUIRED_INPUT_NAME);

		int maxLength = -1;
		try {
			maxLength = Integer.parseInt(maxLengthInputValue);
		} catch (NumberFormatException nfex) {
			PluginUtil.error("FreeTextValidatorPlugin.getJavaScriptFunctionCall(): Unable to get maxLengthInputValue: " + nfex.toString());
		} catch (NullPointerException nex) {
			PluginUtil.error("FreeTextValidatorPlugin.getJavaScriptFunctionCall(): Unable to get maxLengthInputValue: " + nex.toString());
		}

		StringBuffer sb = new StringBuffer();
		sb.append("checkMaxLengthAndRequiredWord(form.");
		sb.append(HtmlUtils.getFreeTextInputName(questionNo));
		sb.append(", ");
		sb.append(maxLength);
		sb.append(", '");
		sb.append(HtmlUtils.htmlEncode(wordRequiredInputValue));
		sb.append("')");
		return sb.toString();
	}


	/**
	 * Gets the javaScriptFunction code for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunction code
	 */
	public String getJavaScriptFunction(int questionNo, HashMap settings) {
		StringBuffer sb = new StringBuffer();
		sb.append("function checkMaxLengthAndRequiredWord(field, maxLength, wordRequired) {\n");
		sb.append("	if(field != null){\n");
		sb.append("		if(field.value != null){\n");
		sb.append("			if(field.value.length > 0){\n");
		sb.append("				if(maxLength > 0 && field.value.length > maxLength){\n");
		sb.append("					return false;\n");
		sb.append("				}\n");
		sb.append("				if(wordRequired != null && field.value.indexOf(wordRequired) < 0){\n");
		sb.append("					return false;\n");
		sb.append("				}\n");
		sb.append("			}\n");
		sb.append("		}\n");
		sb.append("	}\n");
		sb.append("	return true;\n");
		sb.append("}\n");
		return sb.toString();
	}


	/**
	 * Gets a specific setting value
	 *
	 * @param settings Attributes and values settings
	 * @param parameter Parameter name
	 *
	 * @return The javaScriptFunction code
	 */
	private String getSetting(HashMap settings, String parameter) {
		String setting = "";
		if (settings != null) {
			setting = (String) settings.get(parameter);
			if (setting == null) {
				setting = "";
			}
		}
		return setting;
	}
}
