////////////////////////////////////////////////////////////////
// FreeTextRequiredPlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.FreeTextRequiredPlugin;


import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.api.HtmlUtils;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.Question;
import com.objectplanet.survey.plugin.api.QuestionDropdown;
import com.objectplanet.survey.plugin.api.Respondent;
import com.objectplanet.survey.plugin.api.Response;
import com.objectplanet.survey.plugin.api.ResponseDropdown;
import com.objectplanet.survey.plugin.api.ResponseMultiple;
import com.objectplanet.survey.plugin.api.Survey;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.api.SurveySecurityException;
import com.objectplanet.survey.plugin.api.SurveySystemException;
import com.objectplanet.survey.plugin.interfaces.IPluginRequiredValidator;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;


/**
 * Implements FreeTextRequiredPlugin functionality.
 *
 * @author Rikard Halvorsen
 * @created January 25, 2006
 */
public class FreeTextRequiredPlugin extends Plugin implements IPluginRequiredValidator {

	// static values for input fields
	private final static String	ERROR_MSG_INPUT_NAME	= "FreeTextRequiredPlugin_errorMessage";
	private final static String	LAST_CHOICE_INPUT_NAME	= "FreeTextRequiredPlugin_lastChoice";
	private final static String	ALL_SELECTIONS_OPTION	= "1";
	private final static String	LAST_SELECTION_OPTION	= "2";

	// static default value for error message
	private String				ERROR_MSG_DEFAULT		= "Please enter your comment";


	/**
	 * Returns whether the plugin can validate the type of validation, not
	 *
	 * @param customValidatorType The type of custom validation
	 * @return Whether the plugin can validate or not
	 */
	public boolean canValidate(int customValidatorType) {
		return (customValidatorType == CUSTOM_VALIDATOR_QUESTION);
	}


	/**
	 * Validates the question based on the attributes and values set in the settings hashmap
	 *
	 * @param sId Survey id
	 * @param qNo Question number
	 * @param respId Respondent id
	 * @param settings Attributes and values settings
	 * @return Returns if valid or not
	 */
	public boolean validate(long sId, int qNo, long respId, HashMap settings) {
		// ignore the validation if the settings is null
		if (settings == null) {
			return true;
		}

		try {
			if (PluginUtil.isDebugEnabled()) {
				PluginUtil.debug("FreeTextRequiredPlugin.validate() start: s=" + sId + ", qNo=" + qNo + ", r=" + respId);
			}
			
			Survey survey;
			try {
				survey = SurveyManager.getSurvey(PluginUtil.getSystemUser(), sId);
			} catch (SurveySecurityException e1) {
				PluginUtil.error("FreeTextRequiredPlugin.validate(): Could not get survey. (sId " + sId + ", qNo " + qNo + ", respId " + respId + "): " + getStackTraceAsString(e1));
				return true;
			}
			Question question = survey.getQuestion(qNo);
			if (!question.isFreeTextOn()) {
				if (PluginUtil.isDebugEnabled()) {
					PluginUtil.debug("FreeTextRequiredPlugin.validate(): Freetext not enabled from question, exiting. s=" + sId + ", qNo=" + qNo + ", r=" + respId);
				}
				return true;
			}
			
			// get respondent object
			Respondent respondent = SurveyManager.getRespondentForSurvey(respId, sId);

			// get response object for the question
			Response responseToValidate = respondent.getResponse(qNo);

			// get the free text response value
			String freeTextValue = responseToValidate.getFreeText();

			// if freeTextValue is null no free text box in this question
			if (PluginUtil.isDebugEnabled()) {
				PluginUtil.debug("FreeTextRequiredPlugin.validate(): Freetext value: [" + freeTextValue + "], s=" + sId + ", qNo=" + qNo + ", r=" + respId);
			}
			if (freeTextValue == null) {
				return true;
			}

			// if multiple choice question
			if (responseToValidate instanceof ResponseMultiple) {
				ResponseMultiple multipleResponse = (ResponseMultiple) responseToValidate;

				// check the "multiple choice" plugin setting

				String lastChoiceInputValue = getSetting(settings, LAST_CHOICE_INPUT_NAME);
				boolean lastChoiceOnly = lastChoiceInputValue != null && lastChoiceInputValue.equals(LAST_SELECTION_OPTION);

				if (lastChoiceOnly) {
					// validate freetext only if last choice is selected.
					boolean lastChoiceSelected = multipleResponse.getOptionValue(multipleResponse.getOptionValues().length - 1);
					if (!lastChoiceSelected) {
						return true;
					}
				}
			}

			// if dropdown question
			if (responseToValidate instanceof ResponseDropdown) {
				ResponseDropdown dropdownResponse = (ResponseDropdown) responseToValidate;

				// check the "dropdown selection" plugin setting

				String lastChoiceInputValue = getSetting(settings, LAST_CHOICE_INPUT_NAME);
				boolean lastChoiceOnly = lastChoiceInputValue != null && lastChoiceInputValue.equals(LAST_SELECTION_OPTION);

				if (lastChoiceOnly) {
					// validate freetext only if last choice is selected.

					// get the dropdown question
					QuestionDropdown dropdownQuestion = (QuestionDropdown) survey.getQuestion(qNo);

					// get the dropdown
					String[] items = dropdownQuestion.getItems();

					// get the last item
					String lastItem = items[items.length - 1];

					ArrayList selectedValues = dropdownResponse.getSelectedValues();
					boolean lastChoiceSelected = false;
					for (int i = 0; i < selectedValues.size(); i++) {
						String selectedValue = (String) selectedValues.get(i);
						if (selectedValue != null && selectedValue.equals(lastItem)) {
							lastChoiceSelected = true;
						}
					}

					if (!lastChoiceSelected) {
						return true;
					}
				}
			}

			// if free text value is empty, validate false
			if (freeTextValue.trim().equals("")) {
				if (PluginUtil.isDebugEnabled()) {
					PluginUtil.debug("FreeTextRequiredPlugin.validate(): Required freetext not entered. Validation false. Freetext value: [" + freeTextValue + "], s=" + sId + ", qNo=" + qNo + ", r=" + respId);
				}
				return false;
			}
		} catch (SurveySystemException ssex) {
			PluginUtil.error("FreeTextRequiredPlugin.validate(): Unable to validate response (sId " + sId + ", qNo " + qNo + ", respId " + respId + "): " + getStackTraceAsString(ssex));
			// can't get the respondent, we can't validate and return true
		}

		return true;
	}


	/**
	 * Returns the simple HTML setup to be placed in the question edit screen. This code will be
	 * placed within a html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getSimpleHTML(HashMap settings) {
		String errorMsgInputValue = getSetting(settings, ERROR_MSG_INPUT_NAME);

		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr>");
		sb.append("<td width='50%'></td>");
		sb.append("<td class='form label' width='50%'  align=right>");
		sb.append("Custom error message:&nbsp;");
		sb.append("<INPUT TYPE=text name=\"").append(ERROR_MSG_INPUT_NAME);
		sb.append("\" value=\"").append(HtmlUtils.htmlEncode(errorMsgInputValue)).append("\" class=width100>");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Returns the advanced HTML setup to be placed in a popup window when user click on link in the
	 * question edit screen. This code will be placed within an html-TD tag.
	 *
	 * @param settings Attributes and values settings
	 * @return The HTML code
	 */
	public String getAdvancedHTML(HashMap settings) {
		String lastChoiceInputValue = getSetting(settings, LAST_CHOICE_INPUT_NAME);
		String checked1 = "";
		String checked2 = "";
		if (lastChoiceInputValue != null && lastChoiceInputValue.equals("2")) {
			checked2 = "checked";
		} else {
			checked1 = "checked";
		}

		StringBuffer sb = new StringBuffer();
		sb.append("<table width=\"100%\">");
		sb.append("<tr>");
		sb.append("<td width='30%' class='form label'>");
		sb.append("Multiple/dropdown question setting:");
		sb.append("</td>");
		sb.append("<td width='3%' class='form value'>");
		sb.append("<INPUT TYPE=\"radio\" name=\"").append(LAST_CHOICE_INPUT_NAME).append("\"");
		sb.append(" value=\"" + ALL_SELECTIONS_OPTION + "\" " + checked1 + ">");
		sb.append("</td>");
		sb.append("<td width='15%'  class='form label'>");
		sb.append("All selections");
		sb.append("</td>");
		sb.append("<td width='3%' class='form value'>");
		sb.append("<INPUT TYPE=\"radio\" name=\"").append(LAST_CHOICE_INPUT_NAME).append("\"");
		sb.append(" value=\"" + LAST_SELECTION_OPTION + "\" " + checked2 + ">");
		sb.append("</td>");
		sb.append("<td class='form value'>");
		sb.append("Last selection only");
		sb.append("</td>");
		sb.append("</tr>");
		sb.append("</table>");
		return sb.toString();
	}


	/**
	 * Gets the errorMessage to show from the settings hashmap
	 *
	 * @param settings Attributes and values settings
	 * @return The errorMessage value
	 */
	public String getErrorMessage(HashMap settings) {
		String errorMessageToReturn = getSetting(settings, ERROR_MSG_INPUT_NAME);
		if (errorMessageToReturn == null || errorMessageToReturn.equals("")) {
			errorMessageToReturn = ERROR_MSG_DEFAULT;
		}
		return errorMessageToReturn;
	}


	/**
	 * Gets the javaScriptFunctionCall name for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunctionCall name
	 */
	public String getJavaScriptFunctionCall(int questionNo, HashMap settings) {
		StringBuffer sb = new StringBuffer();
		sb.append("checkFreeTextRequired(form.");
		sb.append(HtmlUtils.getFreeTextInputName(questionNo));
		sb.append(")");
		return sb.toString();
	}


	/**
	 * Gets the javaScriptFunction code for client side validation
	 *
	 * @param questionNo Question number
	 * @param settings Attributes and values settings
	 *
	 * @return The javaScriptFunction code
	 */
	public String getJavaScriptFunction(int questionNo, HashMap settings) {
		StringBuffer sb = new StringBuffer();
		sb.append("function checkFreeTextRequired(field) {\n");
		sb.append("	if(field == null || field.value.length == 0){\n");
		sb.append("		return false;\n");
		sb.append("	}\n");
		sb.append("	return true;\n");
		sb.append("}\n");
		return sb.toString();
	}


	/**
	 * Gets a specific setting value. Return empty string if parameter is not set.
	 *
	 * @param settings Attributes and values settings
	 * @param parameter Parameter name
	 *
	 * @return The javaScriptFunction code
	 */
	private String getSetting(HashMap settings, String parameter) {
		if (settings == null) {
			return "";
		}

		String paramValue = (String) settings.get(parameter);
		if (paramValue == null) {
			return "";
		}
		return paramValue;
	}


	/**
	 * Gets the stackTraceAsString attribute
	 *
	 * @param throwable Throwable, containing the stack trace.
	 * @return The stackTraceAsString value
	 */
	private static String getStackTraceAsString(Throwable throwable) {
		ByteArrayOutputStream ostr = new ByteArrayOutputStream();
		throwable.printStackTrace(new PrintStream(ostr));
		return ostr.toString();
	}
}
