////////////////////////////////////////////////////////////////
// EventNotifierPlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.EventNotifierPlugin;


import com.objectplanet.survey.event.Event;
import com.objectplanet.survey.event.EventBusManager;
import com.objectplanet.survey.event.IEventListener;
import com.objectplanet.survey.event.events.AdminEvent;
import com.objectplanet.survey.event.events.NewRespondentEvent;
import com.objectplanet.survey.event.events.NewSurveyEvent;
import com.objectplanet.survey.event.events.NewUserEvent;
import com.objectplanet.survey.event.events.RespondentCompleteEvent;
import com.objectplanet.survey.event.events.SurveyDeletedEvent;
import com.objectplanet.survey.event.events.UserDeletedEvent;
import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.PluginConstants;
import com.objectplanet.survey.plugin.ProcessResult;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.Survey;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.api.SurveySecurityException;
import com.objectplanet.survey.plugin.api.SurveySystemException;
import com.objectplanet.survey.plugin.api.User;
import com.objectplanet.survey.plugin.api.UserManager;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;


/**
 * The event notifier plugin sends email to one recipient, on various events in the survey module.
 * For now, it supports response messages only.
 *
 * @author Torgeir Punnerud
 * @created 13. february 2006
 */
public class EventNotifierPlugin extends Plugin implements IEventListener {

	// Properties props;
	File						propsFile;

	String						newRespondentMessage;
	boolean						newRespondentOn;
	String						newRespondentSubject;
	String						respondentCompleteMessage;
	boolean						respondentCompleteOn;
	String						respondentCompleteSubject;
	boolean						newSurveyEventOn;
	boolean						newUserEventOn;
	boolean						surveyDeletedEventOn;
	boolean						userDeletedEventOn;
	String						toEmail;
	long[]						surveyIds;
	long[]						oldSurveyIds;

	// Default strings and messages messages
	private final static String	PROCESS_NOK							= "<br>The configuration values could <b>not</b> be saved.<br>&nbsp;";
	private final static String	PROCESS_OK							= "<br>The Event Notifier setup is saved.<br>&nbsp;";

	private final static String	SURVEY_NAME							= "[survey name]";
	private final static String	SURVEY_ID							= "[survey id]";
	private final static String	USER_NAME							= "[user name]";
	private final static String	RESPONDENT_ID						= "[respondent id]";

	private final static String	DELETED_USER_NAME					= "[user deleted name]";
	private final static String	DELETED_USER_SUBJ					= "User deleted";
	private final static String	DELETED_USER_MSG					= "A user was deleted \n\nuser: " + DELETED_USER_NAME + "\ndeleted by: " + USER_NAME + "\n";

	private final static String	NEW_RESP_SUBJ						= "New respondent";
	private final static String	NEW_RESP_MSG						= "<br>A new respondent has started to respond to survey \"[survey name]\".\n";

	private final static String	NEW_SURVEY_SUBJ						= "Survey created";
	private final static String	NEW_SURVEY_MSG						= "A survey was created. \n\nsurvey: " + SURVEY_ID + "\ncreated by: " + USER_NAME + "\n";

	private final static String	NEW_USER_NAME						= "[new user name]";
	private final static String	NEW_USER_SUBJ						= "New user created";
	private final static String	NEW_USER_MSG						= "A user was created\n\nNew user: " + NEW_USER_NAME + "\nCreated by: " + USER_NAME + "\n";

	private final static String	RESP_COMPLETE_SUBJ					= "Completed respondent";
	private final static String	RESP_COMPLETE_MSG					= "Respondent has completed \n\nsurvey: \"" + SURVEY_NAME + "\"\nRespondent id: " + RESPONDENT_ID + "\n\n";

	private final static String	SURVEY_DELETED_SUBJ					= "Survey deleted";
	private final static String	SURVEY_DELETED_MSG					= "A survey was deleted \n\nsurvey: " + SURVEY_NAME + "\ndeleted by: " + USER_NAME + "\n";

	// name of the properties
	private static final String	PROP_NEW_RESPONDENT_MESSAGE			= "newRespondentMessage";
	private static final String	PROP_NEW_RESPONDENT_EVENT_ON		= "newRespondentOn";
	private static final String	PROP_RESPONDENT_COMPLETE_MESSAGE	= "respondentCompleteMessage";
	private static final String	PROP_RESPONDENT_COMPLEMTE_EVENT_ON	= "respondentCompleteOn";
	private static final String	PROP_NEW_SURVEY_EVENT_ON			= "newSurveyEventOn";
	private static final String	PROP_NEW_USER_EVENT_ON				= "newUserEventOn";
	private static final String	PROP_SURVEY_DELETE_EVENT_ON			= "surveyDeletedEventOn";
	private static final String	PROP_USER_DELETE_EVENT_ON			= "userDeletedEventOn";
	private static final String	PROP_SURVEY_IDS						= "surveyId";
	private static final String	PROP_TO_EMAIL						= "fromEmail";

	// attribute for validation failure
	private static final String	VALIDATION_FAILED					= "validationFailed";


	/**
	 * Constructor for the EventNotifierPlugin object
	 */
	public EventNotifierPlugin() {
		super();
		propsFile = new File(PluginUtil.getSystemHome() + "/WEB-INF/plugins/EventNotifier.properties");
		newRespondentMessage = NEW_RESP_MSG;
		newRespondentSubject = NEW_RESP_SUBJ;
		respondentCompleteSubject = RESP_COMPLETE_SUBJ;
		respondentCompleteMessage = RESP_COMPLETE_MSG;
		respondentCompleteOn = false;
		newRespondentOn = false;
		newSurveyEventOn = false;
		surveyDeletedEventOn = false;
		newUserEventOn = false;
		userDeletedEventOn = false;
		toEmail = "";
	}


	/**
	 * Returns help html
	 *
	 * @see Plugin#getHelpHTML()
	 */
	public String getHelpHTML() {
		return "The event notifier plugin sends email to one recipient, on various events in the survey module. For now, it supports response messages only.";
	}


	/**
	 * Returns html for the setup page
	 *
	 * @see Plugin#getSetupHTML(HashMap)
	 */
	public String getSetupHTML(HashMap resources) {
		HttpServletRequest request = (HttpServletRequest) resources.get(PluginConstants.RESOURCE_HTTP_REQUEST);
		StringBuffer html = new StringBuffer();

		try {
			// find out if we returned to the setup page after validation failure.
			Boolean validationFailedAttribute = (Boolean) request.getAttribute(VALIDATION_FAILED);
			boolean validationFailed = validationFailedAttribute != null && validationFailedAttribute.booleanValue();

			String formSurveyIds = validationFailed ? request.getParameter("surveyId") : toString(surveyIds);
			String formEmail = validationFailed ? request.getParameter("email") : toEmail;
			String formNewRespMessage = validationFailed ? request.getParameter("newRespondentMessage") : newRespondentMessage;
			String formRespCompleteMessage = validationFailed ? request.getParameter("respondentCompleteMessage") : respondentCompleteMessage;

			boolean formIsNewRespEventOn = isTrue(request.getParameter("newRespondentOn"));
			boolean formIsRespCompleteEventOn = isTrue(request.getParameter("respondentCompleteOn"));
			boolean formIsNewUserEventOn = isTrue(request.getParameter("newUserEventOn"));
			boolean formIsUserDeletedEventOn = isTrue(request.getParameter("userDeletedEventOn"));
			boolean formIsNewSurveyEventOn = isTrue(request.getParameter("newSurveyEventOn"));
			boolean formIsSurveyDeletedEventOn = isTrue(request.getParameter("surveyDeletedEventOn"));

			String formNewRespEventChecked = validationFailed ? getChecked(formIsNewRespEventOn) : getChecked(newRespondentOn);
			String formRespCompleteEventChecked = validationFailed ? getChecked(formIsRespCompleteEventOn) : getChecked(respondentCompleteOn);
			String formNewUserEventChecked = validationFailed ? getChecked(formIsNewUserEventOn) : getChecked(newUserEventOn);
			String formUserDeletedEventChecked = validationFailed ? getChecked(formIsUserDeletedEventOn) : getChecked(userDeletedEventOn);
			String formNewSurveyEventChecked = validationFailed ? getChecked(formIsNewSurveyEventOn) : getChecked(newSurveyEventOn);
			String formSurveyDeletedEventChecked = validationFailed ? getChecked(formIsSurveyDeletedEventOn) : getChecked(surveyDeletedEventOn);

			html.append("<table width=\"100%\">");

			// Survey events
			html.append("	<tr><td class=form colspan=2 height=5></td></tr>");
			html.append("	<tr><td class=sectionTitle colspan=2>Survey events</td></tr>");
			html.append("	<tr><td class=form colspan=2 height=5></td></tr>");
			html.append("	<tr><td class=\"form label\" width=\"30%\"><nobr>Survey ids (separated by comma. 0=all):<nobr></td>");
			html.append("		<td class=\"form value\" width=\"70%\"><INPUT TYPE=\"text\" class=width400 name=\"surveyId\" value=\"").append(formSurveyIds).append("\">");
			html.append("		</td></tr>");

			html.append("	<tr><td class=\"form label\">New respondent notification:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=\"checkbox\" class=checkbox name=\"newRespondentOn\"").append(formNewRespEventChecked).append(">");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"formTop label\">New respondent message:</td>");
			html.append("		<td class=\"form value\"><TEXTAREA class=width400 cols=40 rows=4 name=\"newRespondentMessage\">").append(formNewRespMessage).append("</TEXTAREA>");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"form label\">Respondent complete notification:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=checkbox class=checkbox name=\"respondentCompleteOn\"").append(formRespCompleteEventChecked).append(">");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"formTop label\">Respondent complete message:</td>");
			html.append("		<td class=\"form value\"><TEXTAREA class=width400 cols=40 rows=4 name=\"respondentCompleteMessage\">").append(formRespCompleteMessage).append("</TEXTAREA>");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"form label\">Email address:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=\"text\" class=width400 size=40 name=\"email\" value=\"").append(formEmail).append("\">");
			html.append("		</td></tr>");

			// Admin events
			html.append("	<tr><td class=\"form label\" height=10 colspan=2></td></tr>");
			html.append("	<tr><td class=sectionTitle colspan=2>Admin events</td></tr>");
			html.append("	<tr><td class=form colspan=2 height=5></td></tr>");
			html.append("	<tr><td class=\"form label\">New user notification:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=checkbox class=checkbox name=\"newUserEventOn\"").append(formNewUserEventChecked).append(">");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"form label\">User delete notification:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=checkbox class=checkbox name=\"userDeletedEventOn\"").append(formUserDeletedEventChecked).append(">");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"form label\">New survey notification:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=checkbox class=checkbox name=\"newSurveyEventOn\"").append(formNewSurveyEventChecked).append(">");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"form label\">Survey delete notification:</td>");
			html.append("		<td class=\"form value\"><INPUT TYPE=checkbox class=checkbox name=\"surveyDeletedEventOn\"").append(formSurveyDeletedEventChecked).append(">");
			html.append("		</td></tr>");
			html.append("	<tr><td class=\"form label\" colspan=2 height=5></td></tr>");

			// Save button
			html.append("	<tr><td><td class=\"buttons\"><INPUT TYPE=submit class=button name=submitButton value=\"Save\">");
			html.append("		</td></tr>");
			html.append("</table>");

		} catch (Exception e) {
			e.printStackTrace();
			html.append("Unable to produce html for the plugin setup screen: " + e.getMessage());
		}

		request.setAttribute(VALIDATION_FAILED, Boolean.FALSE);
		return html.toString();
	}


	/**
	 * Method for processing the setup screen. The values from the setup screen is collected by
	 * Opinio and passed to this method in a Map. It is up to the implementor (developer) of this
	 * method to process the values appropriately, and to persist the values if necessary.
	 *
	 * @param values The values entered by the user in the setup screen.
	 * @return The response message, with info about the result of setup values processing.
	 */
	public ProcessResult processSetupHTML(HashMap resources) {
		HttpServletRequest request = (HttpServletRequest) resources.get(PluginConstants.RESOURCE_HTTP_REQUEST);

		// extract values from the request
		String formSurveyIds = request.getParameter("surveyId");
		boolean formIsNewRespEventOn = isTrue(request.getParameter("newRespondentOn"));
		String formNewRespMessage = request.getParameter("newRespondentMessage");
		boolean formIsRespCompleteEventOn = isTrue(request.getParameter("respondentCompleteOn"));
		String formRespCompleteMessage = request.getParameter("respondentCompleteMessage");
		boolean formIsNewUserEventOn = isTrue(request.getParameter("newUserEventOn"));
		boolean formIsUserDeletedEventOn = isTrue(request.getParameter("userDeletedEventOn"));
		boolean formIsNewSurveyEventOn = isTrue(request.getParameter("newSurveyEventOn"));
		boolean formIsSurveyDeletedEventOn = isTrue(request.getParameter("surveyDeletedEventOn"));
		String formEmail = request.getParameter("email");

		// Validate input: validate survey ids
		if (!surveyIdsValid(formSurveyIds)) {
			request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
			return new ProcessResult("Invalid survey ids", true);
		}

		// Validate input: "New respondent message" must be set if "New respondent notification" is
		// on
		if (formIsNewRespEventOn) {
			if (formNewRespMessage == null || formNewRespMessage.trim().equals("")) {
				request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
				return new ProcessResult("Please enter the message for \"New respondent event\"", true);
			}
		}
		// Validate input: "Respondent completed message" must be set if "Respondent completed
		// notification" is on
		if (formIsRespCompleteEventOn) {
			if (formRespCompleteMessage == null || formRespCompleteMessage.trim().equals("")) {
				request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
				return new ProcessResult("Please enter the message for \"Respondent complete event\"", true);
			}
		}

		// Validate input: email address must be set
		if (formEmail == null || formEmail.trim().equals("")) {
			request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
			return new ProcessResult("Please enter your email address", true);
		}

		// all entered values are corrent. Store the values as local fields for faster access.
		newRespondentMessage = formNewRespMessage;
		newRespondentOn = formIsNewRespEventOn;
		respondentCompleteMessage = formRespCompleteMessage;
		respondentCompleteOn = formIsRespCompleteEventOn;
		newSurveyEventOn = formIsNewSurveyEventOn;
		newUserEventOn = formIsNewUserEventOn;
		surveyDeletedEventOn = formIsSurveyDeletedEventOn;
		userDeletedEventOn = formIsUserDeletedEventOn;
		surveyIds = extractSurveyIds(formSurveyIds);
		toEmail = formEmail;

		// Store the values in the property file (make them persistent)
		Properties props = new Properties();
		props.put(PROP_NEW_RESPONDENT_MESSAGE, formNewRespMessage);
		props.put(PROP_NEW_RESPONDENT_EVENT_ON, formIsNewRespEventOn + "");
		props.put(PROP_RESPONDENT_COMPLETE_MESSAGE, formRespCompleteMessage);
		props.put(PROP_RESPONDENT_COMPLEMTE_EVENT_ON, formIsRespCompleteEventOn + "");
		props.put(PROP_NEW_SURVEY_EVENT_ON, formIsNewSurveyEventOn + "");
		props.put(PROP_NEW_USER_EVENT_ON, formIsNewUserEventOn + "");
		props.put(PROP_SURVEY_DELETE_EVENT_ON, formIsSurveyDeletedEventOn + "");
		props.put(PROP_USER_DELETE_EVENT_ON, formIsUserDeletedEventOn + "");
		props.put(PROP_SURVEY_IDS, formSurveyIds);
		props.put(PROP_TO_EMAIL, formEmail);
		try {
			PluginUtil.storeProps(props, propsFile);
		} catch (IOException e) {
			String error = PROCESS_NOK;
			error = error + "<br><br>" + e.getMessage();
			ProcessResult result = new ProcessResult(error, true);
			request.setAttribute(VALIDATION_FAILED, Boolean.TRUE);
			return result;
		}

		// register the listeners
		registerListeners(true);

		// remember the old survey ids, so that if user changes them later, the listeners to the
		// removed surveys will be unregistered.
		oldSurveyIds = surveyIds;
		return new ProcessResult(PROCESS_OK, false);
	}


	/**
	 * Handles the events that this plugin is registered to receive.
	 *
	 * @see IEventListener#handleEvent(Event)
	 * @param event The event.
	 */
	public void handleEvent(Event event) {

		if (event instanceof NewRespondentEvent && newRespondentOn) {

			NewRespondentEvent nrEvent = (NewRespondentEvent) event;
			sendEmail(NEW_RESP_SUBJ, prepareSurveyMessage(newRespondentMessage, nrEvent.getSurveyId(), nrEvent.getRespondentId()));

		} else if (event instanceof RespondentCompleteEvent && respondentCompleteOn) {

			RespondentCompleteEvent rcEvent = (RespondentCompleteEvent) event;
			sendEmail(RESP_COMPLETE_SUBJ, prepareSurveyMessage(respondentCompleteMessage, rcEvent.getSurveyId(), rcEvent.getRespondentId()));

		} else if (event instanceof NewSurveyEvent && newSurveyEventOn) {

			NewSurveyEvent nsEvent = (NewSurveyEvent) event;
			sendEmail(NEW_SURVEY_SUBJ, getAdminMessage(nsEvent));

		} else if (event instanceof SurveyDeletedEvent && surveyDeletedEventOn) {

			SurveyDeletedEvent sdEvent = (SurveyDeletedEvent) event;
			sendEmail(SURVEY_DELETED_SUBJ, getAdminMessage(sdEvent));

		} else if (event instanceof NewUserEvent && newUserEventOn) {

			NewUserEvent nuEvent = (NewUserEvent) event;
			sendEmail(NEW_USER_SUBJ, getAdminMessage(nuEvent));

		} else if (event instanceof UserDeletedEvent && userDeletedEventOn) {

			UserDeletedEvent udEvent = (UserDeletedEvent) event;
			sendEmail(DELETED_USER_SUBJ, getAdminMessage(udEvent));

		}
	}


	/**
	 * Starts the plugin. The survey event bus registrations are made here. Any defaults, plus
	 * whatever event settings are made through the setup screen are read here.
	 *
	 * @see Plugin#start()
	 */
	public void start() {
		try {
			// load the properties
			FileInputStream in = new FileInputStream(propsFile);
			BufferedInputStream buf = new BufferedInputStream((InputStream) in);
			Properties props = new Properties();
			props.load(buf);

			// copy properties to local fields for faster access
			newRespondentMessage = (String) props.get(PROP_NEW_RESPONDENT_MESSAGE);
			newRespondentOn = isTrue((String) props.get(PROP_NEW_RESPONDENT_EVENT_ON));
			respondentCompleteMessage = (String) props.get(PROP_RESPONDENT_COMPLETE_MESSAGE);
			respondentCompleteOn = isTrue((String) props.get(PROP_RESPONDENT_COMPLEMTE_EVENT_ON));
			newSurveyEventOn = isTrue((String) props.get(PROP_NEW_SURVEY_EVENT_ON));
			newUserEventOn = isTrue((String) props.get(PROP_NEW_USER_EVENT_ON));
			surveyDeletedEventOn = isTrue((String) props.get(PROP_SURVEY_DELETE_EVENT_ON));
			userDeletedEventOn = isTrue((String) props.get(PROP_USER_DELETE_EVENT_ON));
			surveyIds = extractSurveyIds((String) props.get(PROP_SURVEY_IDS));
			oldSurveyIds = surveyIds;
			toEmail = (String) props.get(PROP_TO_EMAIL);
			in.close();
		} catch (FileNotFoundException fe) {
			PluginUtil.warn(propsFile.getAbsolutePath() + " file not found: " + fe.toString());
			return;
		} catch (IOException ie) {
			PluginUtil.error(propsFile.getAbsolutePath() + "/EventNotifier.properties file could not be loaded: " + ie.getMessage());
			return;
		}

		// register the listeners
		registerListeners(false);

		// log the plugin status
		PluginUtil.info("EventNotifierPlugin loaded");

	}


	/**
	 * Parse the passed string object and return list of Long objects
	 *
	 * @param string The string with numbers, comma separated.
	 * @return The longValues list of Long objects
	 */
	private long[] extractSurveyIds(String formSurveyIds) {
		if (formSurveyIds == null || formSurveyIds.equals("")) {
			return new long[0];
		}

		String[] stringValues = formSurveyIds.split(" ?, ?");
		long[] longValues = new long[stringValues.length];

		for (int i = 0; i < stringValues.length; i++) {
			try {
				longValues[i] = Long.parseLong(stringValues[i]);
			} catch (NumberFormatException e) {
			}
		}
		return longValues;
	}


	/**
	 * Validate survey ids (comma separated numbers)
	 *
	 * @param formSurveyIds
	 * @return
	 */
	private boolean surveyIdsValid(String formSurveyIds) {
		// check if survey ids contains ONLY numbers, commars and spaces
		Pattern numersPattern = Pattern.compile("[\\d, ]+");
		Matcher matcher = numersPattern.matcher(formSurveyIds);
		if (!matcher.matches()) {
			return false;
		}
		return true;
	}


	/**
	 * Get boolean value from the string object
	 *
	 * @param request
	 * @param paramName
	 * @return
	 */
	private boolean isTrue(String param) {
		if (param == null) {
			return false;
		}
		if (param.equalsIgnoreCase("true") || param.equalsIgnoreCase("on")) {
			return true;
		}
		return false;
	}


	/**
	 * Register listeners
	 */
	private void registerListeners(boolean removeOldListeners) {

		EventBusManager eventMgr = EventBusManager.instance();

		if (removeOldListeners) {
			eventMgr.removeListener(this, EventBusManager.BUS_TYPE_USER);
			eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY_MANAGEMENT);

			// unregister any registered id's
			if (oldSurveyIds != null) {
				if (oldSurveyIds.length == 1 && oldSurveyIds[0] == 0) {
					eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_NEW_RESPONDENT);
					eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_PRE_SURVEY_PAGE);
					eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_DISPLAY_QUESTION);
					eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_COMPLETE);
					eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_DATA_FOUND);
				} else {
					for (int i = 0; i < oldSurveyIds.length; i++) {
						long surveyId = oldSurveyIds[i];
						eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_NEW_RESPONDENT, surveyId);
						eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_PRE_SURVEY_PAGE, surveyId);
						eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_DISPLAY_QUESTION, surveyId);
						eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_COMPLETE, surveyId);
						eventMgr.removeListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_DATA_FOUND, surveyId);
					}
				}
			}
		}

		if (newUserEventOn || userDeletedEventOn) {
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_USER);
		}
		if (newSurveyEventOn || surveyDeletedEventOn) {
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY_MANAGEMENT);
		}

		// if 0 is set as survey id - register to listen to all new surveys
		if (surveyIds.length == 1 && surveyIds[0] == 0) {
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_NEW_RESPONDENT);
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_PRE_SURVEY_PAGE);
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_DISPLAY_QUESTION);
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_COMPLETE);
			eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_DATA_FOUND);
		} else {

			// register to listen to new surveys with the specified ids
			for (int i = 0; i < surveyIds.length; i++) {
				long surveyId = surveyIds[i];
				eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_NEW_RESPONDENT, surveyId);
				eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_PRE_SURVEY_PAGE, surveyId);
				eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_DISPLAY_QUESTION, surveyId);
				eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_COMPLETE, surveyId);
				eventMgr.addListener(this, EventBusManager.BUS_TYPE_SURVEY, EventBusManager.EVENT_TYPE_RESPONDENT_DATA_FOUND, surveyId);
			}
		}
	}


	/**
	 * Prepares an new survey message for sending email.
	 *
	 * @param event Description of the Parameter
	 * @return The message, prepared.
	 */
	private String getAdminMessage(AdminEvent event) {
		String message = "";
		User user = getUser(event.getUserId());
		Survey survey = null;
		long surveyId;
		StringBuffer buf = null;
		String findString;
		String replaceString;

		if (event instanceof NewSurveyEvent) {
			message = NEW_SURVEY_MSG;
			NewSurveyEvent nEvent = (NewSurveyEvent) event;
			message = prepareAdminMessage(message, user);
			// insert survey name
			findString = SURVEY_ID;
			surveyId = nEvent.getSurveyId();
			buf = new StringBuffer(message);
			replaceString = surveyId + "";
			replace(findString, replaceString, buf);
		} else if (event instanceof SurveyDeletedEvent) {
			message = SURVEY_DELETED_MSG;
			SurveyDeletedEvent sEvent = (SurveyDeletedEvent) event;
			message = prepareAdminMessage(message, user);
			// insert survey id
			findString = SURVEY_NAME;
			surveyId = sEvent.getSurveyId();
			try {
				survey = SurveyManager.getSurvey(PluginUtil.getSystemUser(), surveyId);
				replaceString = survey.getSurveyName();
				buf = new StringBuffer(message);
				replace(findString, replaceString, buf);
			} catch (SurveySystemException e) {
				return message;
			} catch (SurveySecurityException e) {
				return message;
			}
		} else if (event instanceof NewUserEvent) {
			message = NEW_USER_MSG;
			NewUserEvent nEvent = (NewUserEvent) event;
			message = prepareAdminMessage(message, user);
			// insert survey id
			findString = NEW_USER_NAME;
			replaceString = getLogin(nEvent.getNewUserId());

			buf = new StringBuffer(message);
			replace(findString, replaceString, buf);
		} else if (event instanceof UserDeletedEvent) {
			message = DELETED_USER_MSG;
			buf = new StringBuffer(message);
			UserDeletedEvent dEvent = (UserDeletedEvent) event;
			message = prepareAdminMessage(message, user);
			// insert survey id
			findString = USER_NAME;
			replaceString = user.getLogin();
			replace(findString, replaceString, buf);

			findString = DELETED_USER_NAME;
			replaceString = dEvent.getDeletedUserLogin();
			replace(findString, replaceString, buf);
		}

		if (buf != null) {
			return buf.toString();
		} else {
			return message;
		}
	}


	/**
	 * Long array to String with comma separated numbers
	 *
	 * @param longArray
	 * @return
	 */
	private String toString(long[] longArray) {
		if (longArray == null) {
			return "";
		}

		StringBuffer out = new StringBuffer();
		for (int i = 0; i < longArray.length; i++) {
			if (i > 0) {
				out.append(", ");
			}
			out.append(longArray[i]);
		}
		return out.toString();
	}


	/**
	 * Gets the checked attribute for a checkbox element. Returns text "checked" or "null" based on
	 * the passed value.
	 *
	 * @param checked text to attach to the html checkbox
	 * @return The checked text
	 */
	private String getChecked(boolean checked) {
		if (checked) {
			return " checked";
		} else {
			return "";
		}
	}


	/**
	 * Notification messages contain survey names, respondent ids etc. These are dynamic, and needs
	 * to be inserted into the messages, for every message sent.
	 *
	 * @param message The message to prepare
	 * @param surveyId Survey id
	 * @param respondentId Respondent id
	 * @return The message, prepared.
	 */
	private String prepareSurveyMessage(String message, long surveyId, long respondentId) {
		if (message == null) {
			return null;
		}

		StringBuffer buf = new StringBuffer(message);
		try {
			Survey survey = SurveyManager.getSurvey(PluginUtil.getSystemUser(), surveyId);
			replace(SURVEY_NAME, survey.getSurveyName(), buf);
			replace(RESPONDENT_ID, "" + respondentId, buf);
			return buf.toString();
		} catch (SurveySystemException e) {
			return message;
		} catch (SurveySecurityException e) {
			return message;
		}
	}


	/**
	 * Gets user object. Returns null if exception occurs.
	 */
	private User getUser(long userId) {
		try {
			User user = UserManager.getUser(PluginUtil.getSystemUser(), userId);
			return user;
		} catch (SurveySystemException e) {
			return null;
		} catch (SurveySecurityException e) {
			return null;
		}
	}


	/**
	 * Gets user object. Returns null if exception occurs.
	 */
	private String getLogin(long userId) {
		User user = getUser(userId);
		if (user != null) {
			return user.getLogin();
		}
		return "Unknown";
	}


	/**
	 * Prepares an admin message for sending email. The user name is inserted into the message, if
	 * it contains the string USER_NAME (constant).
	 *
	 * @param message The message to prepare
	 * @param user User that performed the action
	 * @return The message, prepared.
	 */
	private String prepareAdminMessage(String message, User user) {
		if (message == null || user == null) {
			return message;
		}

		StringBuffer buf = new StringBuffer(message);
		String findString;
		String replaceString = "Unknown";

		// insert user name in message
		findString = USER_NAME;
		if (user != null) {
			replaceString = user.getLogin();
		}
		replace(findString, replaceString, buf);

		return buf.toString();
	}


	/**
	 * Replaces all occurrences of a string with another string in a string buffer.
	 *
	 * @param s String to search for
	 * @param s2 String to replace with
	 * @param buf The string buffer to perform the search/replace operation on.
	 */
	private void replace(String s, String s2, StringBuffer buf) {
		int length = s.length();
		int position;
		String bufString = buf.toString();
		position = bufString.indexOf(s, 0);

		while (position > -1) {
			buf.replace(position, position + length, s2);
			bufString = buf.toString();
			position = bufString.indexOf(s, position + length);
		}
	}


	/**
	 * Sends the notification about the events registered.
	 *
	 * @param subject Subject of the notification
	 * @param message The message to send
	 */
	private void sendEmail(String subject, String message) {
		// PENDING: make the From email and name configurable
		String fromName = "Opinio";
		String fromEmail = toEmail;
		try {
			PluginUtil.sendEmail(fromName, fromEmail, fromEmail, subject, message, "text/html");
		} catch (Exception e) {
			PluginUtil.error(e.toString());
		}
	}

}
