////////////////////////////////////////////////////////////////
// CSVExportExamplePlugin.java
//
// Copyright (C) ObjectPlanet, Inc.
// All rights reserved.
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.plugin.survey.CSVExportExamplePlugin;


import java.io.File;
import java.sql.SQLException;
import java.util.HashMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.exception.ExceptionUtils;

import com.objectplanet.survey.event.CallBack;
import com.objectplanet.survey.plugin.Plugin;
import com.objectplanet.survey.plugin.PluginConstants;
import com.objectplanet.survey.plugin.ProcessResult;
import com.objectplanet.survey.plugin.api.CSVExportParameters;
import com.objectplanet.survey.plugin.api.PluginUtil;
import com.objectplanet.survey.plugin.api.Survey;
import com.objectplanet.survey.plugin.api.SurveyManager;
import com.objectplanet.survey.plugin.api.SurveySystemException;
import com.objectplanet.survey.plugin.api.User;
import com.objectplanet.survey.plugin.api.UserManager;
import com.objectplanet.survey.plugin.interfaces.menu.IPageActionSurvey;


/**
 * A simple plugin to demonstrate CSV export via the Plugin API.
 * 
 * @author Torgeir Lund
 * @created 21 September 2015 02:24 PM
 */
public class CSVExportExamplePlugin extends Plugin implements IPageActionSurvey {

	private static final String	PARAM_NAME_TO_RESPONDENT_ID		= "toRespondentId";
	private static final String	PARAM_NAME_FROM_RESPONDENT_ID	= "fromRespondentId";

	private static final String	PARAM_NAME_FROM_START_DATE		= "fromStartDate";
	private static final String	PARAM_NAME_TO_START_DATE		= "toStartDate";
	
	private static final String	PARAM_NAME_FROM_COMPLETED_DATE	= "fromCompletedDate";
	private static final String	PARAM_NAME_TO_COMPLETED_DATE	= "toCompletedDate";

	// ------ CONSTANTS ------------------
	public static final String	SESSION_ATTR_USER_ID			= "USER_ID";

	private static final String	PARAM_NAME_SURVEY_ID			= "surveyId";
	private static final String	PARAM_NAME_REPORT_ID			= "reportId";
	private static final String	PARAM_NAME_REPORT_CUSTOM_ID		= "reportIdCustom";
	private static final String	PARAM_NAME_ATTRIBUTES			= "attributes";
	private static final String	PARAM_NAME_INVITEE_EMAIL		= "inviteeEmail";
	private static final String	PARAM_NAME_INVITEE_NAME			= "inviteeName";
	private static final String	PARAM_NAME_INVITEE_ID			= "inviteeId";
	private static final String	PARAM_NAME_FILENAME				= "filename";


	// ----------- class variables -----------------

	/**
	 * Constructor for the CSVExportExamplePlugin object
	 */
	public CSVExportExamplePlugin() {
		super();
	}


	/**
	 * Starts the plugin. Initialize the properties set for the plugin.
	 */
	public void start() {
		// get the properties file and store the values in a local properties variable.

		// register events

		PluginUtil.info(getClassName() + " loaded");
	}


	/**
	 * Function labels.
	 */
	public String getFunctionLabel(int functionNo, HashMap resources) {
		return "Export to CSV";
	}


	/**
	 * get html for the hosting account page
	 */
	public String getFunctionHTML(int functionNo, HashMap resources) {
		try {
			// retrieve request and session objects
			HttpServletRequest request = (HttpServletRequest) resources.get(PluginConstants.RESOURCE_HTTP_REQUEST);
			HttpSession session = request.getSession();

			// get current user id & view user id
			// long currentUserId = ((Long) session.getAttribute(SESSION_ATTR_USER_ID)).longValue();
			long surveyId = Long.parseLong(request.getParameter(PARAM_NAME_SURVEY_ID));
			String getResportsSql = "SELECT ReportId FROM OPS_Report WHERE SurveyId = ?";

			LongList reportIds = DatabaseUtils.loadIds(null, getResportsSql, surveyId, 5);
			String surveyName;
			try {
				surveyName = DatabaseUtils.getString(null, "SELECT SurveyName FROM OPS_Survey WHERE SurveyId = ?", surveyId);
			} catch (SQLException e1) {
				return "Error getting survey name: " + e1.getMessage();
			}

			StringBuffer buf = new StringBuffer();
			buf.append("<div class='form label' style='padding: 15px'>Current survey: <b>" + surveyName + "</b></div>");
			buf.append("<div style='padding: 15px; padding-bottom: 0px'>Export survey to file:</div>");
			buf.append("<div style='padding: 15px; padding-top: 0px'>");
			buf.append("	<input type='text' name='filename' id='filename' style='width: 500px' value='c:\\data.zip' />");
			buf.append("</div>");

			buf.append("<div style='padding: 15px; padding-bottom: 0px'>Attributes to include (comma separated):</div>");
			buf.append("<div style='padding: 15px; padding-top: 0px'>");
			buf.append("	<input type='text' name='attributes' id='attributes' style='width: 500px' value='' />");
			buf.append("</div>");

			buf.append("<div style='padding: 15px'>");
			buf.append("	<label style='margin-right: 20px'><input type='checkbox' name='inviteeId' value='true' />Invitee id</label>");
			buf.append("	<label style='margin-right: 20px'><input type='checkbox' name='inviteeName' value='true' />Invitee name</label>");
			buf.append("	<label style='margin-right: 20px'><input type='checkbox' name='inviteeEmail' value='true' />Invitee email</label>");
			buf.append("</div>");

			buf.append("<div style='padding: 15px; padding-bottom: 0px'>Report:</div>");
			buf.append("<div style='padding: 15px; padding-top: 0px'>");
			buf.append("	<SELECT name='reportId' style='width: 200px'>");
			buf.append("		<OPTION value=''>Select a report (optional)</OPTION>");
			for (int i = 0; i < reportIds.size(); i++) {
				buf.append("		<OPTION value='" + reportIds.get(i) + "'>" + reportIds.get(i) + "</OPTION>");
			}
			buf.append("	</SELECT>");
			buf.append("	<input type='text' name='" + PARAM_NAME_REPORT_CUSTOM_ID + "' style='width: 150px' placeholder='report id' value='' /> ");
			buf.append("</div>");

			buf.append("<div style='padding: 15px; padding-bottom: 0px'>Respondents to include (from-to IDs, empty/0-value disables). Overrides dates below</div>");
			buf.append("<div style='padding: 15px; padding-top: 0px'>");
			buf.append("	<input type='text' name='" + PARAM_NAME_FROM_RESPONDENT_ID + "' style='width: 150px' placeholder='from-id, not inclusive' value='1' /> to <input type='text' name='" + PARAM_NAME_TO_RESPONDENT_ID + "' style='width: 150px' placeholder='to-id, inclusive' value='99999999' /> ");
			buf.append("</div>");

			buf.append("<div style='padding: 15px; padding-bottom: 0px'>Response start dates (milliseconds since 1970, empty/0-value disables). Overides completed dates below.</div>");
			buf.append("<div style='padding: 15px; padding-top: 0px'>");
			buf.append("	<input type='text' name='" + PARAM_NAME_FROM_START_DATE + "' style='width: 150px' placeholder='from start date, inclusive' value='0' /> to <input type='text' name='" + PARAM_NAME_TO_START_DATE + "' style='width: 150px' placeholder='to start date, inclusive' value='0' /> ");
			buf.append("</div>");

			buf.append("<div style='padding: 15px; padding-bottom: 0px'>Response completed dates (milliseconds since 1970, empty/0-value disables):</div>");
			buf.append("<div style='padding: 15px; padding-top: 0px'>");
			buf.append("	<input type='text' name='" + PARAM_NAME_FROM_COMPLETED_DATE + "' style='width: 150px' placeholder='from completed date, inclusive' value='0' /> to <input type='text' name='" + PARAM_NAME_TO_COMPLETED_DATE + "' style='width: 150px' placeholder='to completed date, inclusive' value='0' /> ");
			buf.append("</div>");

			buf.append("<div style='padding: 15px'><button style='padding: 4px;' id='exportButton'>Start CSV export</button></div>");

			return buf.toString();

		} catch (Exception e) {
			PluginUtil.error("CSVExportExamplePlugin: Error when producing function html: " + e.getMessage());
			e.printStackTrace();
			return "CSVExportExamplePlugin: Error when producing function html: " + e.getMessage();
		}
	}


	/**
	 * Get the parameters, and start the export
	 */
	public ProcessResult processFunctionHTML(int functionNo, HashMap resources) {
		PluginUtil.debug("CSVExportExamplePluginUtil.processFunctionHTML(): " + functionNo);
		try {
			HttpServletRequest request = (HttpServletRequest) resources.get(PluginConstants.RESOURCE_HTTP_REQUEST);
			long surveyId = Long.parseLong(request.getParameter(PARAM_NAME_SURVEY_ID));

			String reportIdStr = request.getParameter(PARAM_NAME_REPORT_ID);
			String reportIdCustomStr = request.getParameter(PARAM_NAME_REPORT_CUSTOM_ID);
			long reportId = 0;
			if (!PluginUtil.isNullOrEmpty(reportIdStr)) {
				reportId = Long.parseLong(reportIdStr);
			} else {
				reportId = (!PluginUtil.isNullOrEmpty(reportIdCustomStr)) ? Long.parseLong(reportIdCustomStr) : 0;
			}

			User user = PluginUtil.getUserFromSession(request);
			String fileName = request.getParameter(PARAM_NAME_FILENAME);
			File exportFile = null;
			String settingsStr = null;
			try {
				if (!UserManager.hasSurveyPermission(PluginUtil.getSystemUser(), user.getUserId(), surveyId, UserManager.PERMISSION_READ)) {
					return new ProcessResult("User " + user.getLogin() + " do not have sufficient access rights to perform a CSV export for survey " + surveyId + ".", false);
				}

				Survey survey = SurveyManager.getSurvey(user, surveyId);
				if (survey.getCompletedRespondentCount() < 1) {
					return new ProcessResult("There are no completed respondents to export.", false);
				}

				// create a callback object to handle a callback.
				MyCallBack callBackObj = new MyCallBack(surveyId);

				// export all (presuming no respondents exist with ID greater than 99999999)
				String defaultFileName = PluginUtil.isWindows() ? "c:\\temp\\CSV_from_plugin.zip" : "/tmp/CSV_from_plugin.zip";
				exportFile = new File(fileName == null ? defaultFileName : fileName);

				CSVExportParameters params = new CSVExportParameters(surveyId, exportFile);
				params.setReportId(reportId);
				
				params.setFromRespondentId(getRequestLongValue(request, PARAM_NAME_FROM_RESPONDENT_ID, 0));
				params.setToRespondentId(getRequestLongValue(request, PARAM_NAME_TO_RESPONDENT_ID, 0));

				// set start/completed date ranges
				params.setFromStartDate(getRequestLongValue(request, PARAM_NAME_FROM_START_DATE, 0));
				params.setToStartDate(getRequestLongValue(request, PARAM_NAME_TO_START_DATE, 0));
				params.setFromCompletedDate(getRequestLongValue(request, PARAM_NAME_FROM_COMPLETED_DATE, 0));
				params.setToCompletedDate(getRequestLongValue(request, PARAM_NAME_TO_COMPLETED_DATE, 0));

				params.setCallBackObj(callBackObj);

				// include/exclude various attributes
				String attributes = request.getParameter(PARAM_NAME_ATTRIBUTES);
				if (!PluginUtil.isNullOrEmpty(attributes)) {
					params.setRespondentAttributesIncluded(true);
					params.setRespondentAttributesToIncludeInExport(attributes.split(","));
				}
				params.setInviteeIdIncluded("true".equals(request.getParameter(PARAM_NAME_INVITEE_ID)));
				params.setInviteeNameIncluded("true".equals(request.getParameter(PARAM_NAME_INVITEE_NAME)));
				params.setInviteeEmailIncluded("true".equals(request.getParameter(PARAM_NAME_INVITEE_EMAIL)));

				// perform the actual CSV export. It will be started in a thread, and this method
				// will return immediately.
				SurveyManager.exportCSV(params);

				settingsStr = "<div style='padding-top: 20px;'>Attributes: " + attributes + "</div><div>Invite id: " + params.isInviteeIdIncluded() + "</div><div>Invitee name: " + params.isInviteeNameIncluded() + "</div><div>Invitee email: " + params.isInviteeEmailIncluded() + "</div><div>Report: " + reportId + "</div><div>Respondents: " + params.getFromRespondentId() + " (exclusive) to " + params.getToRespondentId() + " (inclusive)</div><div>Response start dates: " + params.getFromStartDate() + " (inclusive) to " + params.getToStartDate() + " (inclusive)</div><div>Response completed dates: " + params.getFromCompletedDate() + " (inclusive) to " + params.getToCompletedDate() + " (inclusive)</div>";
			} catch (SurveySystemException e) {
				PluginUtil.error("CSVExportExamplePluginUtil.processFunctionHTML(): " + e.getMessage());
				return new ProcessResult("CSVExportExamplePluginUtil.processFunctionHTML(): Unable to start export process. Error has occurred. " + e.getMessage(), true);
			}
			return new ProcessResult("<div style='margin-bottom: 20px'>CSV export for survey " + surveyId + " has started successfully.</div>" + settingsStr + "<div>File: " + exportFile.getAbsolutePath() + "</div>", false);
		} catch (Exception e) {
			PluginUtil.error("CSVExportExamplePlugin.getFunctionHTML(): " + e.getMessage() + ExceptionUtils.getStackTrace(e));
			return new ProcessResult("Error occurred while processing plugin function: " + e.getMessage(), true);
		}
	}


	/**
	 * Gets an number from the request object, and return it as int
	 * 
	 * @param request The request object
	 * @param param Name of request parameter
	 * @param defaultValue Default value, if parameter is not found
	 * @return The int value
	 */
	public static int getRequestIntValue(HttpServletRequest request, String param, int defaultValue) {
		String paramValue = request.getParameter(param);
		if (paramValue == null) {
			return defaultValue;
		}
		try {
			return (new Integer(paramValue)).intValue();
		} catch (NumberFormatException ne) {
			return defaultValue;
		}
	}


	/**
	 * Gets an number from the request object, and return it a long
	 * 
	 * @param request The request object
	 * @param param Name of request parameter
	 * @param defaultValue Default value, if parameter is not found
	 * @return The long value
	 */
	public static long getRequestLongValue(HttpServletRequest request, String param, long defaultValue) {
		String paramValue = request.getParameter(param);
		if (paramValue == null) {
			return defaultValue;
		}
		try {
			return (new Long(paramValue)).longValue();
		} catch (NumberFormatException ne) {
			return defaultValue;
		}
	}
	
	
	/**
	 * Returns the plugin's class name.
	 * 
	 * @return The className value
	 */
	public String getClassName() {
		return getClass().getName();
	}

	// a simple callback handler for the export
	public class MyCallBack implements CallBack {

		long	surveyId;


		public MyCallBack(long surveyId) {
			this.surveyId = surveyId;
		}


		public void callBack() {
			PluginUtil.info("CSVExportExamplePlugin: export for survey " + getSurveyId() + " is complete.");
		}


		public void callBackOnError(String errorInfo) {
			PluginUtil.info("CSVExportExamplePlugin: export for survey " + getSurveyId() + " FAILED. " + errorInfo);
		}


		public long getSurveyId() {
			return surveyId;
		}
	};

}
